/*
Copyright 2013 Adobe Systems Incorporated.  All rights reserved. 

Purpose:
Initialization of LiveEdit Feature and Liveview Bridging

*/
/*jslint vars: true, plusplus: true, devel: true, browser: true, nomen: true, maxerr: 50 */
/*global GenericLiveEditBridger,TextHudLiveViewBridger,ExtensionsBridgeFunctions,dw,DWfile,DWLE_FILEPATHS, DW_LIVEEDIT_CONSTANTS, DW_ICE_HEADLIGHTS*/

function LiveEditBridgingObject(browser, documentDOM) {
    /*
        Member variables : browser, browser object window and dreamweaver configuration path 
    */
    'use strict';
    this.m_browser = browser;
    this.m_context = this;
    this.m_browserWindow = browser.getWindow();
    this.m_liveConfigPath = dw.getConfigurationPath() + DWLE_FILEPATHS.LiveEditPath;
    this.m_TextHudLEBridger = new TextHudLiveViewBridger(documentDOM);
    this.m_ExtensionsBridgeFunctions = new ExtensionsBridgeFunctions(documentDOM);
	this.m_dom = documentDOM;

    // Is it DW template instance
    var theDOM = documentDOM;
    this.liveEditTemplateInstance = false;
    if (theDOM && theDOM.isTemplateInstance()) {
        this.liveEditTemplateInstance = true;
    }
}

/*
    Two major steps
    1. attachDwFunctions : attaching dw js APIS to browser, so that we can call them from browser
    2. evaluateLiveEditJS : evalJs on browser all the required scripts that we need.
    
    Note: Please keep your functions organized.
    Utility functions that can be used across all the huds should be kept at begining.
*/

/*
        This function attaches DW functionalitties to browser
*/
LiveEditBridgingObject.prototype.attachConstantsToBrowser = function () {

    'use strict';
    if (!this.m_browserWindow) {
        return;
    }
    /*
            these are the constants that are need on browser instance.
    */

    this.m_browserWindow.liveEditResourcePath = this.m_liveConfigPath;
    // all live edit resources are kept under this folder.

    this.m_browserWindow.liveEditOverlayCss = DWfile.read(this.m_liveConfigPath + DWLE_FILEPATHS.OverLayCss);
    // this is css we need for the overlay (etc orange border for text editing )    

    // Set Editable Region's CSS only for template instance.
    if (this.liveEditTemplateInstance) {
        this.m_browserWindow.liveEditTemplateInstance = true;
        this.m_browserWindow.liveEditableRegionCss = DWfile.read(this.m_liveConfigPath + DWLE_FILEPATHS.EditableRegionCss);
    }
    
    this.m_browserWindow.liveEditFGInstance = false;
    this.m_browserWindow.doctype = "";
    if (this.m_dom && this.m_dom.isFluidGridPreviewCefMode()) {
        this.m_browserWindow.liveEditFGInstance = true;
    }
    if (this.m_dom) {
        this.m_browserWindow.doctype = this.m_dom.documentType;
    }
        
   
    //read extensions details    
    this.m_browserWindow.extensionsDetail = this.readExtensionsDetail();
    this.m_browserWindow.extensionsResourcePath = this.m_liveConfigPath + DWLE_FILEPATHS.ExtensionsFolder;  // pass the path to the browser to load other files as required

    this.m_browserWindow.appLanguage = dw.getAppLanguage();
};

/*
    readExtensionsDetail - read deatils about the extensions to be loaded from config file and return the array of extensions
    details : path to the html file to be loaded, id of extension
*/
LiveEditBridgingObject.prototype.readExtensionsDetail = function () {
    'use strict';
    var i;
    var extensionsXml = dw.getDocumentDOM(this.m_liveConfigPath + DWLE_FILEPATHS.LiveViewExtensionsListFile, false);
    var liveviewExtensions = [];
    if (extensionsXml) {
        var extensions = extensionsXml.getElementsByTagName('extension');
        for (i = 0; i < extensions.length; i++) {
            var src = extensions[i].getAttribute("src");
            var id = extensions[i].getAttribute("id");
            var type = extensions[i].getAttribute("type");
            if (src && id) {
                var extensionEntry = {};
                extensionEntry.src = src;
                extensionEntry.id = id;
                if (type) {
                    extensionEntry.type = type;
                    if (type === "auxiliary") {
                        var tags = extensions[i].getAttribute("tags");
                        if (tags) {
                            extensionEntry.tags = tags;
                        }
                    }
                }
                liveviewExtensions[i] = extensionEntry;
            }
        }
    }
    return liveviewExtensions;
};

/*
    logDebugMsg - for SM layer to log messages in log file.
*/

/*
        This function attaches DW functionalitties to browser
*/
LiveEditBridgingObject.prototype.attachDwFunctions = function () {
    'use strict';
    if (!this.m_browserWindow) {
        return;
    }
    var thisContext = this;

	// Initialize Log
	var writeDWLog = function (content) {
        if (DW_LIVEEDIT_DEBUGGING === true) {
            var dwUserConfigPath = dw.getUserConfigurationPath();
            DWfile.write(dwUserConfigPath + "/liveEditLog.txt", new Date() + "         " + content + "\r\n", "append");
        }
    };
	this.logDebugMsg = writeDWLog;
    this.m_browserWindow.writeDWLog = writeDWLog;

    this.m_browserWindow.DWLECallJsBridgingFunction = function (type, fnName, args, stopEditOp) {

		if (!thisContext.m_dom) {
			thisContext.logDebugMsg('ERROR - ******NO DOM or DOM is already in EDIT ******* dom = ' + thisContext.m_dom);
			return;
		}

		// if we are not in LiveView when we get the bridge call
		// We should not proceed further.
		if (thisContext.m_dom.getDesignViewMode() !== "live") {
			thisContext.logDebugMsg('ERROR - Not in Live View when recieved Bridge call = ' + fnName);
			return;
		}

		var currentBridge = null;
        try {
            if (stopEditOp === true) {
                // Mark this DOM update as Live Edit change.
                dw.setLiveEditBasedEditOpInProgress(true);
            }

			// Choose the bridge.
            if (type === DW_LIVEEDIT_CONSTANTS.HudTextEdit) {
				currentBridge = thisContext.m_TextHudLEBridger;
            } else if (type === DW_LIVEEDIT_CONSTANTS.Extensions) {
				currentBridge = thisContext.m_ExtensionsBridgeFunctions;
            }

			// Call the bridge function.
			if (currentBridge) {
				currentBridge[fnName](args);
			}
		} catch (e) {
			thisContext.logDebugMsg('ERROR message : ' + e.message);
			thisContext.logDebugMsg('ERROR Info : ' + e.stack);
			if (stopEditOp) {
				dw.showInfoBarMessage('liveview/infobar/liveEditCommitFailed', 'warning', 'Editable Live View');
				if (thisContext.m_browser) {
					thisContext.m_browser.cancelLiveViewOperationState();
				}
				if (currentBridge) {
					currentBridge.rollBackFailedChanges();
				}
				if (thisContext.m_browser) {
					thisContext.m_browser.refreshPage();
				}
                dw.logEvent(DW_ICE_HEADLIGHTS.OTH_ELV, DW_ICE_HEADLIGHTS.CommitFailed);
            }
        } finally {
            // Done with Live Edit change.
            dw.setLiveEditBasedEditOpInProgress(false);
        }

    };
};

/*
    this function eval's our javascript on live view browser.
*/

LiveEditBridgingObject.prototype.evaluateLiveEditJS = function () {
    'use strict';
    if (!this.m_browser || !this.m_browserWindow) {
        return;
    }
    var jquery = DWfile.read(this.m_liveConfigPath + "/jquery-1.8.3.js");
    this.m_browser.evalJS(jquery);
    //make our own jQuery instance for use so that it wont conflict with user's jQuery
    //we should use '$dwJQuery' instead of '$' in our jQuery plugins
    this.m_browser.evalJS('var $dwJQuery = jQuery.noConflict(true);');
    var i18nPluginJs = DWfile.read(this.m_liveConfigPath + "/jquery.i18n.properties-1.0.9.js");
    this.m_browser.evalJS(i18nPluginJs);
    var constantsJs = DWfile.read(this.m_liveConfigPath + "/Constants.js");
    var genericHudJs = DWfile.read(this.m_liveConfigPath + "/GenericHud.js");
    var liveEditEventJs = DWfile.read(this.m_liveConfigPath + "/LiveEditEvent.js");
    var textEditJs = DWfile.read(this.m_liveConfigPath + "/textEdit.js");
    var controllerJs = DWfile.read(this.m_liveConfigPath + "/controller.js");
    var dwExtensionControllerJs = DWfile.read(this.m_liveConfigPath + "/Extensions/dwUtility.js");
    /*
        This Order needs to be maintained.
        
        Any generic codes that apply to all huds should be kept at the begining (i.e. like genericHud.js and eventTarget.js)
        
        Individual Hud implementations should be kept next.
        
        controller should be the last one to be eval'ed.
    */
    /*
        GenericHud.JS --> 
            definition of abstract GenericHUDObject Object. any functionalities common to all HUDS should be member functions of this class.

        eventTarget.JS -->
            provides event listening and firing mechanisms.
        
        Following files contains definition of all the HUDS which are derived from GenericHudObject defined in GenericHud.js
        textEdit.Js --> text Editing Hud functionalities
        
        
        Controller.js
        This controller is responsible for event delegation to each of the huds and error handling.
        any event to the Huds will go through controller ONLY*.
    */

    this.m_browser.evalJS(constantsJs);
    this.m_browser.evalJS(liveEditEventJs);
    this.m_browser.evalJS(dwExtensionControllerJs);
    this.m_browser.evalJS(genericHudJs);
    this.m_browser.evalJS(textEditJs);
    // Eval Editable region javascript before Controller execution.
    if (this.liveEditTemplateInstance) {
        var editableRgnJs = DWfile.read(this.m_liveConfigPath + "/TemplateInstanceEditableRgn.js");
        this.m_browser.evalJS(editableRgnJs);
    }
    this.m_browser.evalJS(controllerJs);

    var partialRefreshHTMLSimpleDOM = DWfile.read(this.m_liveConfigPath + "/PartialRefresh/HTMLSimpleDOM.js");
    var partialRefreshHTMLDOMDiff = DWfile.read(this.m_liveConfigPath + "/PartialRefresh/HTMLDOMDiff.js");
    var partialRefreshMurmurHash = DWfile.read(this.m_liveConfigPath + "/PartialRefresh/murmurhash3_gc.js");
    var partialRefreshRemoteFunctions = DWfile.read(this.m_liveConfigPath + "/PartialRefresh/RemoteFunctions.js");
    var partialRefreshHTMLTokenizer = DWfile.read(this.m_liveConfigPath + "/PartialRefresh/HTMLTokenizer.js");
    var partialRefreshLiveUpdateTools = DWfile.read(this.m_liveConfigPath + "/PartialRefresh/LiveUpdateTools.js");

    this.m_browser.evalJS(partialRefreshHTMLSimpleDOM);
    this.m_browser.evalJS(partialRefreshHTMLDOMDiff);
    this.m_browser.evalJS(partialRefreshMurmurHash);
    this.m_browser.evalJS(partialRefreshRemoteFunctions);
    this.m_browser.evalJS(partialRefreshHTMLTokenizer);
    this.m_browser.evalJS(partialRefreshLiveUpdateTools);

    //load extensions infrastructure
    var extensionsJs = DWfile.read(this.m_liveConfigPath + "/InitLiveViewExtensionsInfrastructure.js");
    this.m_browser.evalJS(extensionsJs);
};
